import { Navbar } from "@/components/navbar"
import { Footer } from "@/components/footer"
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Users, Calendar, Clock, Video, MessageSquare, PlusCircle, Search } from "lucide-react"
import Image from "next/image"
import Link from "next/link"

export default function LiveStudy() {
  // Sample study rooms
  const studyRooms = [
    {
      id: 1,
      title: "Advanced Calculus Study Group",
      subject: "Mathematics",
      participants: 8,
      maxParticipants: 12,
      scheduledTime: "Today, 3:00 PM",
      duration: "2 hours",
      host: "Prof. David Chen",
      hostAvatar: "/images/avatar-prof.jpg",
      image: "/images/calculus-study.jpg",
      tags: ["Calculus", "Mathematics", "Group Study"],
    },
    {
      id: 2,
      title: "Physics Problem Solving Session",
      subject: "Physics",
      participants: 5,
      maxParticipants: 10,
      scheduledTime: "Today, 5:30 PM",
      duration: "1.5 hours",
      host: "Sarah Johnson",
      hostAvatar: "/images/avatar-1.jpg",
      image: "/images/physics-study.jpg",
      tags: ["Physics", "Problem Solving", "Mechanics"],
    },
    {
      id: 3,
      title: "Literature Analysis: Shakespeare",
      subject: "Literature",
      participants: 6,
      maxParticipants: 15,
      scheduledTime: "Tomorrow, 4:00 PM",
      duration: "2 hours",
      host: "Emma Lewis",
      hostAvatar: "/images/avatar-5.jpg",
      image: "/images/literature-study.jpg",
      tags: ["Literature", "Shakespeare", "Analysis"],
    },
    {
      id: 4,
      title: "Chemistry Exam Prep",
      subject: "Chemistry",
      participants: 12,
      maxParticipants: 20,
      scheduledTime: "Tomorrow, 6:00 PM",
      duration: "3 hours",
      host: "Dr. Michael Brown",
      hostAvatar: "/images/avatar-4.jpg",
      image: "/images/chemistry-study.jpg",
      tags: ["Chemistry", "Exam Prep", "Organic Chemistry"],
    },
    {
      id: 5,
      title: "Computer Science: Algorithms",
      subject: "Computer Science",
      participants: 7,
      maxParticipants: 15,
      scheduledTime: "Friday, 5:00 PM",
      duration: "2 hours",
      host: "Carlos Martinez",
      hostAvatar: "/images/avatar-6.jpg",
      image: "/images/cs-study.jpg",
      tags: ["Computer Science", "Algorithms", "Data Structures"],
    },
    {
      id: 6,
      title: "Biology Study Session",
      subject: "Biology",
      participants: 9,
      maxParticipants: 15,
      scheduledTime: "Saturday, 10:00 AM",
      duration: "2.5 hours",
      host: "Dr. Lisa Wong",
      hostAvatar: "/images/avatar-3.jpg",
      image: "/images/biology-study.jpg",
      tags: ["Biology", "Genetics", "Cell Biology"],
    },
  ]

  return (
    <div className="min-h-screen flex flex-col">
      <Navbar />

      <main className="flex-grow">
        <div className="bg-teal-500 text-white py-12 relative overflow-hidden">
          <div className="absolute inset-0 z-0">
            <Image
              src="https://hebbkx1anhila5yf.public.blob.vercel-storage.com/Live%20Study%20Cyberspace-DzZs9tldaBYR33ceSUelcs7YKP91Zl.jpeg"
              alt="Live Study Cyberspace Background"
              fill
              className="object-cover opacity-20"
            />
          </div>
          <div className="container mx-auto px-4 relative z-10">
            <h1 className="text-3xl md:text-4xl font-bold mb-4">Live Study Cyberspace</h1>
            <p className="text-xl max-w-3xl">
              Join real-time study sessions, collaborate with peers, and enhance your learning through interactive
              discussions.
            </p>
          </div>
        </div>

        <div className="container mx-auto px-4 py-8">
          <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4 mb-8">
            <div className="relative w-full md:w-96">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" size={18} />
              <Input placeholder="Search study rooms..." className="pl-10" />
            </div>

            <Button className="bg-teal-500 hover:bg-teal-600">
              <PlusCircle className="mr-2 h-4 w-4" />
              Create Study Room
            </Button>
          </div>

          <Tabs defaultValue="all" className="mb-8">
            <TabsList>
              <TabsTrigger value="all">All Rooms</TabsTrigger>
              <TabsTrigger value="active">Active Now</TabsTrigger>
              <TabsTrigger value="scheduled">Scheduled</TabsTrigger>
              <TabsTrigger value="my-rooms">My Rooms</TabsTrigger>
            </TabsList>

            <TabsContent value="all" className="mt-6">
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {studyRooms.map((room) => (
                  <Card key={room.id} className="overflow-hidden hover:shadow-lg transition-shadow">
                    <div className="relative h-48">
                      <Image src={room.image || "/placeholder.svg"} alt={room.title} fill className="object-cover" />
                      <div className="absolute top-2 right-2 bg-white/90 text-teal-800 px-2 py-1 rounded text-xs font-medium">
                        {room.subject}
                      </div>
                    </div>
                    <CardHeader>
                      <CardTitle className="text-lg">{room.title}</CardTitle>
                      <CardDescription>Hosted by {room.host}</CardDescription>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center text-gray-500">
                          <Users className="h-4 w-4 mr-1" />
                          <span className="text-sm">
                            {room.participants}/{room.maxParticipants} participants
                          </span>
                        </div>
                        <div className="flex">
                          <Avatar className="h-6 w-6 border-2 border-white">
                            <AvatarImage src={room.hostAvatar || "/placeholder.svg"} alt={room.host} />
                            <AvatarFallback>{room.host.charAt(0)}</AvatarFallback>
                          </Avatar>
                          {/* Would show more avatars here */}
                        </div>
                      </div>

                      <div className="flex items-center gap-4">
                        <div className="flex items-center text-gray-500">
                          <Calendar className="h-4 w-4 mr-1" />
                          <span className="text-sm">{room.scheduledTime}</span>
                        </div>
                        <div className="flex items-center text-gray-500">
                          <Clock className="h-4 w-4 mr-1" />
                          <span className="text-sm">{room.duration}</span>
                        </div>
                      </div>

                      <div className="flex flex-wrap gap-2">
                        {room.tags.map((tag, index) => (
                          <span key={index} className="text-xs bg-teal-100 text-teal-800 px-2 py-1 rounded-full">
                            {tag}
                          </span>
                        ))}
                      </div>
                    </CardContent>
                    <CardFooter className="flex gap-2">
                      <Button asChild className="flex-1 bg-teal-500 hover:bg-teal-600">
                        <Link href={`/live-study/${room.id}`}>
                          <MessageSquare className="h-4 w-4 mr-2" />
                          Join Chat
                        </Link>
                      </Button>
                      <Button variant="outline" className="flex-1">
                        <Video className="h-4 w-4 mr-2" />
                        Join Video
                      </Button>
                    </CardFooter>
                  </Card>
                ))}
              </div>
            </TabsContent>

            <TabsContent value="active">
              <div className="py-12 text-center text-gray-500">Sign in to view currently active study rooms</div>
            </TabsContent>

            <TabsContent value="scheduled">
              <div className="py-12 text-center text-gray-500">Sign in to view upcoming scheduled study sessions</div>
            </TabsContent>

            <TabsContent value="my-rooms">
              <div className="py-12 text-center text-gray-500">Sign in to view your study rooms</div>
            </TabsContent>
          </Tabs>

          <div className="bg-gray-50 rounded-xl p-6 mt-12">
            <div className="flex flex-col md:flex-row items-start md:items-center gap-6">
              <div className="relative w-full md:w-1/3 h-48 rounded-lg overflow-hidden">
                <Image src="/images/virtual-classroom.jpg" alt="Virtual Classroom" fill className="object-cover" />
              </div>
              <div className="md:w-2/3">
                <h2 className="text-2xl font-bold mb-2">Host Your Own Study Session</h2>
                <p className="text-gray-600 mb-4">
                  Create a virtual study room, invite classmates, and lead discussions on topics you're passionate
                  about. Share resources, collaborate on problems, and help each other succeed.
                </p>
                <Button className="bg-teal-500 hover:bg-teal-600">Create a Study Room</Button>
              </div>
            </div>
          </div>
        </div>
      </main>

      <Footer />
    </div>
  )
}
