import { Navbar } from "@/components/navbar"
import { Footer } from "@/components/footer"
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card"
import { Progress } from "@/components/ui/progress"
import { Button } from "@/components/ui/button"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { BookOpen, Clock } from "lucide-react"
import Image from "next/image"
import Link from "next/link"

export default function PracticeLabs() {
  // Sample subject data
  const subjects = [
    {
      id: 1,
      name: "Physics",
      description: "Mechanics, thermodynamics, electricity, and more",
      image: "/images/physics-lab.jpg",
      quizCount: 24,
      progress: 45,
    },
    {
      id: 2,
      name: "Mathematics",
      description: "Algebra, calculus, geometry, and statistics",
      image: "/images/math-lab.jpg",
      quizCount: 32,
      progress: 68,
    },
    {
      id: 3,
      name: "Chemistry",
      description: "Organic, inorganic, and physical chemistry",
      image: "/images/chemistry-lab.jpg",
      quizCount: 18,
      progress: 30,
    },
    {
      id: 4,
      name: "Biology",
      description: "Cell biology, genetics, ecology, and evolution",
      image: "/images/biology-lab.jpg",
      quizCount: 22,
      progress: 15,
    },
    {
      id: 5,
      name: "Computer Science",
      description: "Programming, algorithms, data structures",
      image: "/images/cs-lab.jpg",
      quizCount: 28,
      progress: 52,
    },
    {
      id: 6,
      name: "Literature",
      description: "Poetry, prose, drama, and literary analysis",
      image: "/images/literature-lab.jpg",
      quizCount: 20,
      progress: 40,
    },
  ]

  // Sample recent quizzes
  const recentQuizzes = [
    {
      id: 1,
      title: "Newton's Laws of Motion",
      subject: "Physics",
      questions: 15,
      timeEstimate: "20 min",
      image: "/images/physics-quiz.jpg",
    },
    {
      id: 2,
      title: "Quadratic Equations",
      subject: "Mathematics",
      questions: 12,
      timeEstimate: "15 min",
      image: "/images/math-quiz.jpg",
    },
    {
      id: 3,
      title: "Periodic Table Elements",
      subject: "Chemistry",
      questions: 20,
      timeEstimate: "25 min",
      image: "/images/chemistry-quiz.jpg",
    },
  ]

  return (
    <div className="min-h-screen flex flex-col">
      <Navbar />

      <main className="flex-grow">
        <div className="bg-teal-500 text-white py-12 relative overflow-hidden">
          <div className="absolute inset-0 z-0">
            <Image
              src="https://hebbkx1anhila5yf.public.blob.vercel-storage.com/Practice%20Labs-MKdKuignIEMrL3hD2ZBqg45iE4h7LL.png"
              alt="Practice Labs Background"
              fill
              className="object-cover opacity-20"
            />
          </div>
          <div className="container mx-auto px-4 relative z-10">
            <h1 className="text-3xl md:text-4xl font-bold mb-4">Practice Labs</h1>
            <p className="text-xl max-w-3xl">
              Enhance your learning with interactive quizzes and exercises across various subjects.
            </p>
          </div>
        </div>

        <div className="container mx-auto px-4 py-8">
          <Tabs defaultValue="subjects" className="mb-8">
            <TabsList>
              <TabsTrigger value="subjects">Subject Labs</TabsTrigger>
              <TabsTrigger value="recent">Recent Quizzes</TabsTrigger>
              <TabsTrigger value="recommended">Recommended</TabsTrigger>
              <TabsTrigger value="completed">Completed</TabsTrigger>
            </TabsList>

            <TabsContent value="subjects" className="mt-6">
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {subjects.map((subject) => (
                  <Card key={subject.id} className="overflow-hidden hover:shadow-lg transition-shadow">
                    <div className="relative h-48">
                      <Image
                        src={subject.image || "/placeholder.svg"}
                        alt={subject.name}
                        fill
                        className="object-cover"
                      />
                    </div>
                    <CardHeader>
                      <CardTitle>{subject.name}</CardTitle>
                      <CardDescription>{subject.description}</CardDescription>
                    </CardHeader>
                    <CardContent>
                      <div className="flex items-center justify-between mb-2">
                        <span className="text-sm text-gray-500">{subject.quizCount} quizzes</span>
                        <span className="text-sm font-medium">{subject.progress}% complete</span>
                      </div>
                      <Progress value={subject.progress} className="h-2" />
                    </CardContent>
                    <CardFooter>
                      <Button asChild className="w-full bg-teal-500 hover:bg-teal-600">
                        <Link href={`/practice-labs/${subject.id}`}>Enter Lab</Link>
                      </Button>
                    </CardFooter>
                  </Card>
                ))}
              </div>
            </TabsContent>

            <TabsContent value="recent" className="mt-6">
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {recentQuizzes.map((quiz) => (
                  <Card key={quiz.id} className="overflow-hidden hover:shadow-lg transition-shadow">
                    <div className="relative h-48">
                      <Image src={quiz.image || "/placeholder.svg"} alt={quiz.title} fill className="object-cover" />
                      <div className="absolute top-2 right-2 bg-white/90 text-teal-800 px-2 py-1 rounded text-xs font-medium">
                        {quiz.subject}
                      </div>
                    </div>
                    <CardHeader>
                      <CardTitle className="text-lg">{quiz.title}</CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="flex items-center gap-4">
                        <div className="flex items-center text-gray-500">
                          <BookOpen className="h-4 w-4 mr-1" />
                          <span className="text-sm">{quiz.questions} questions</span>
                        </div>
                        <div className="flex items-center text-gray-500">
                          <Clock className="h-4 w-4 mr-1" />
                          <span className="text-sm">{quiz.timeEstimate}</span>
                        </div>
                      </div>
                    </CardContent>
                    <CardFooter>
                      <Button asChild className="w-full bg-teal-500 hover:bg-teal-600">
                        <Link href={`/practice-labs/quiz/${quiz.id}`}>Start Quiz</Link>
                      </Button>
                    </CardFooter>
                  </Card>
                ))}
              </div>
            </TabsContent>

            <TabsContent value="recommended">
              <div className="py-12 text-center text-gray-500">Sign in to view personalized quiz recommendations</div>
            </TabsContent>

            <TabsContent value="completed">
              <div className="py-12 text-center text-gray-500">Sign in to view your completed quizzes and progress</div>
            </TabsContent>
          </Tabs>

          <div className="bg-gray-50 rounded-xl p-6 mt-12">
            <div className="flex flex-col md:flex-row items-start md:items-center gap-6">
              <div className="relative w-full md:w-1/3 h-48 rounded-lg overflow-hidden">
                <Image src="/images/study-group.jpg" alt="Study Group" fill className="object-cover" />
              </div>
              <div className="md:w-2/3">
                <h2 className="text-2xl font-bold mb-2">Join a Study Group</h2>
                <p className="text-gray-600 mb-4">
                  Connect with fellow students to practice together, discuss challenging topics, and improve your
                  understanding through collaboration.
                </p>
                <div className="flex flex-wrap gap-4">
                  <Button className="bg-teal-500 hover:bg-teal-600">Find Study Groups</Button>
                  <Button variant="outline">Create a Group</Button>
                </div>
              </div>
            </div>
          </div>
        </div>
      </main>

      <Footer />
    </div>
  )
}
