import { Navbar } from "@/components/navbar"
import { Footer } from "@/components/footer"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Progress } from "@/components/ui/progress"
import { Badge } from "@/components/ui/badge"
import { Trophy, BookOpen, Users, Lightbulb, Target } from "lucide-react"
import Image from "next/image"

export default function ActivityProfile() {
  const userStats = {
    name: "Alex Johnson",
    avatar: "/images/avatar-default.jpg",
    level: 15,
    points: 2450,
    nextLevelPoints: 2800,
    joinDate: "September 2023",
    badges: [
      { name: "Idea Innovator", icon: "💡", description: "Posted 10+ innovative ideas" },
      { name: "Quiz Master", icon: "🧠", description: "Completed 50+ quizzes" },
      { name: "Collaborator", icon: "🤝", description: "Joined 5+ project teams" },
      { name: "Study Leader", icon: "📚", description: "Led 3+ study sessions" },
    ],
    recentActivities: [
      { type: "idea", title: "Posted 'AI Physics Simulator'", points: 50, date: "2 days ago" },
      { type: "quiz", title: "Completed Calculus Quiz", points: 25, date: "3 days ago" },
      { type: "project", title: "Joined 'Sustainable Energy' project", points: 30, date: "1 week ago" },
      { type: "study", title: "Attended Physics Study Session", points: 20, date: "1 week ago" },
    ],
    subjects: [
      { name: "Physics", progress: 85, quizzes: 24, rank: "Advanced" },
      { name: "Mathematics", progress: 72, quizzes: 18, rank: "Intermediate" },
      { name: "Computer Science", progress: 90, quizzes: 32, rank: "Expert" },
      { name: "Chemistry", progress: 45, quizzes: 12, rank: "Beginner" },
    ],
  }

  return (
    <div className="min-h-screen flex flex-col">
      <Navbar />

      <main className="flex-grow">
        <div className="bg-teal-500 text-white py-12 relative overflow-hidden">
          <div className="absolute inset-0 z-0">
            <Image
              src="https://hebbkx1anhila5yf.public.blob.vercel-storage.com/Activity%20Profiles.jpg-p5JKRPWozNIIGuOtQ2WzIH6gBdfbKb.jpeg"
              alt="Activity Profile Background"
              fill
              className="object-cover opacity-20"
            />
          </div>
          <div className="container mx-auto px-4 relative z-10">
            <h1 className="text-3xl md:text-4xl font-bold mb-4">Activity Profile</h1>
            <p className="text-xl max-w-3xl">
              Track your learning journey, achievements, and progress across all Dolabing activities.
            </p>
          </div>
        </div>

        <div className="container mx-auto px-4 py-8">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <div className="lg:col-span-1">
              <Card className="mb-6">
                <CardHeader className="text-center">
                  <Avatar className="h-24 w-24 mx-auto mb-4">
                    <AvatarImage src={userStats.avatar || "/placeholder.svg"} alt={userStats.name} />
                    <AvatarFallback className="text-2xl">{userStats.name.charAt(0)}</AvatarFallback>
                  </Avatar>
                  <CardTitle className="text-2xl">{userStats.name}</CardTitle>
                  <CardDescription>Member since {userStats.joinDate}</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="text-center">
                    <div className="text-3xl font-bold text-teal-600">{userStats.points}</div>
                    <div className="text-sm text-gray-500">Total Points</div>
                  </div>

                  <div>
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-sm font-medium">Level {userStats.level}</span>
                      <span className="text-sm text-gray-500">
                        {userStats.points}/{userStats.nextLevelPoints}
                      </span>
                    </div>
                    <Progress value={(userStats.points / userStats.nextLevelPoints) * 100} className="h-3" />
                    <div className="text-xs text-gray-500 mt-1">
                      {userStats.nextLevelPoints - userStats.points} points to next level
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center">
                    <Trophy className="h-5 w-5 mr-2 text-yellow-500" />
                    Achievements
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-2 gap-3">
                    {userStats.badges.map((badge, index) => (
                      <div key={index} className="text-center p-3 bg-gray-50 rounded-lg">
                        <div className="text-2xl mb-1">{badge.icon}</div>
                        <div className="text-xs font-medium">{badge.name}</div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </div>

            <div className="lg:col-span-2">
              <Tabs defaultValue="overview" className="mb-8">
                <TabsList className="grid w-full grid-cols-4">
                  <TabsTrigger value="overview">Overview</TabsTrigger>
                  <TabsTrigger value="subjects">Subjects</TabsTrigger>
                  <TabsTrigger value="activities">Activities</TabsTrigger>
                  <TabsTrigger value="achievements">Achievements</TabsTrigger>
                </TabsList>

                <TabsContent value="overview" className="mt-6">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                    <Card>
                      <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Ideas Posted</CardTitle>
                        <Lightbulb className="h-4 w-4 text-muted-foreground" />
                      </CardHeader>
                      <CardContent>
                        <div className="text-2xl font-bold">12</div>
                        <p className="text-xs text-muted-foreground">+2 from last month</p>
                      </CardContent>
                    </Card>

                    <Card>
                      <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Quizzes Completed</CardTitle>
                        <BookOpen className="h-4 w-4 text-muted-foreground" />
                      </CardHeader>
                      <CardContent>
                        <div className="text-2xl font-bold">86</div>
                        <p className="text-xs text-muted-foreground">+12 from last month</p>
                      </CardContent>
                    </Card>

                    <Card>
                      <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Study Sessions</CardTitle>
                        <Users className="h-4 w-4 text-muted-foreground" />
                      </CardHeader>
                      <CardContent>
                        <div className="text-2xl font-bold">24</div>
                        <p className="text-xs text-muted-foreground">+5 from last month</p>
                      </CardContent>
                    </Card>

                    <Card>
                      <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Projects Joined</CardTitle>
                        <Target className="h-4 w-4 text-muted-foreground" />
                      </CardHeader>
                      <CardContent>
                        <div className="text-2xl font-bold">7</div>
                        <p className="text-xs text-muted-foreground">+1 from last month</p>
                      </CardContent>
                    </Card>
                  </div>

                  <Card>
                    <CardHeader>
                      <CardTitle>Recent Activities</CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="space-y-4">
                        {userStats.recentActivities.map((activity, index) => (
                          <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                            <div>
                              <div className="font-medium">{activity.title}</div>
                              <div className="text-sm text-gray-500">{activity.date}</div>
                            </div>
                            <Badge variant="secondary">+{activity.points} pts</Badge>
                          </div>
                        ))}
                      </div>
                    </CardContent>
                  </Card>
                </TabsContent>

                <TabsContent value="subjects" className="mt-6">
                  <div className="space-y-4">
                    {userStats.subjects.map((subject, index) => (
                      <Card key={index}>
                        <CardContent className="p-6">
                          <div className="flex items-center justify-between mb-4">
                            <div>
                              <h3 className="font-bold text-lg">{subject.name}</h3>
                              <p className="text-sm text-gray-500">{subject.quizzes} quizzes completed</p>
                            </div>
                            <Badge variant={subject.rank === "Expert" ? "default" : "secondary"}>{subject.rank}</Badge>
                          </div>
                          <div className="space-y-2">
                            <div className="flex justify-between text-sm">
                              <span>Progress</span>
                              <span>{subject.progress}%</span>
                            </div>
                            <Progress value={subject.progress} className="h-2" />
                          </div>
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                </TabsContent>

                <TabsContent value="activities">
                  <div className="py-12 text-center text-gray-500">Detailed activity timeline coming soon...</div>
                </TabsContent>

                <TabsContent value="achievements">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    {userStats.badges.map((badge, index) => (
                      <Card key={index}>
                        <CardContent className="p-6 text-center">
                          <div className="text-4xl mb-2">{badge.icon}</div>
                          <h3 className="font-bold mb-1">{badge.name}</h3>
                          <p className="text-sm text-gray-600">{badge.description}</p>
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                </TabsContent>
              </Tabs>
            </div>
          </div>
        </div>
      </main>

      <Footer />
    </div>
  )
}
