"use client"

import type React from "react"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Send, Smile, Paperclip } from "lucide-react"

interface ChatMessage {
  id: number
  user: string
  avatar: string
  message: string
  timestamp: string
  type: "text" | "image" | "file"
}

interface ChatInterfaceProps {
  roomId: string
}

export function ChatInterface({ roomId }: ChatInterfaceProps) {
  const [message, setMessage] = useState("")
  const [messages, setMessages] = useState<ChatMessage[]>([
    {
      id: 1,
      user: "Prof. David Chen",
      avatar: "/images/avatar-prof.jpg",
      message: "Welcome everyone! Let's start with integration by substitution. Any questions from the reading?",
      timestamp: "2:30 PM",
      type: "text",
    },
    {
      id: 2,
      user: "Sarah Johnson",
      avatar: "/images/avatar-1.jpg",
      message: "I'm having trouble with the u-substitution method. Could you walk through an example?",
      timestamp: "2:32 PM",
      type: "text",
    },
    {
      id: 3,
      user: "Miguel Rodriguez",
      avatar: "/images/avatar-2.jpg",
      message: "Same here! Especially when the substitution isn't obvious.",
      timestamp: "2:33 PM",
      type: "text",
    },
    {
      id: 4,
      user: "Prof. David Chen",
      avatar: "/images/avatar-prof.jpg",
      message:
        "Great questions! Let me share a whiteboard example. We'll work through ∫(2x+1)√(x²+x+3)dx step by step.",
      timestamp: "2:35 PM",
      type: "text",
    },
    {
      id: 5,
      user: "Carlos Martinez",
      avatar: "/images/avatar-6.jpg",
      message: "This is really helpful! I can see the pattern now.",
      timestamp: "2:40 PM",
      type: "text",
    },
  ])

  const handleSendMessage = () => {
    if (message.trim()) {
      const newMessage: ChatMessage = {
        id: messages.length + 1,
        user: "You",
        avatar: "/images/avatar-default.jpg",
        message: message.trim(),
        timestamp: new Date().toLocaleTimeString([], { hour: "2-digit", minute: "2-digit" }),
        type: "text",
      }
      setMessages([...messages, newMessage])
      setMessage("")
    }
  }

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault()
      handleSendMessage()
    }
  }

  return (
    <Card className="h-96">
      <CardHeader>
        <CardTitle className="text-lg">Study Room Chat</CardTitle>
      </CardHeader>
      <CardContent className="p-0 flex flex-col h-full">
        <div className="flex-1 overflow-y-auto p-4 space-y-4">
          {messages.map((msg) => (
            <div key={msg.id} className="flex items-start space-x-3">
              <Avatar className="h-8 w-8">
                <AvatarImage src={msg.avatar || "/placeholder.svg"} alt={msg.user} />
                <AvatarFallback>{msg.user.charAt(0)}</AvatarFallback>
              </Avatar>
              <div className="flex-1">
                <div className="flex items-center space-x-2 mb-1">
                  <span className="text-sm font-medium">{msg.user}</span>
                  <span className="text-xs text-gray-500">{msg.timestamp}</span>
                </div>
                <div className="bg-gray-100 rounded-lg p-3 text-sm">{msg.message}</div>
              </div>
            </div>
          ))}
        </div>

        <div className="border-t p-4">
          <div className="flex items-center space-x-2">
            <Input
              value={message}
              onChange={(e) => setMessage(e.target.value)}
              onKeyPress={handleKeyPress}
              placeholder="Type your message..."
              className="flex-1"
            />
            <Button variant="ghost" size="sm">
              <Paperclip className="h-4 w-4" />
            </Button>
            <Button variant="ghost" size="sm">
              <Smile className="h-4 w-4" />
            </Button>
            <Button onClick={handleSendMessage} size="sm" className="bg-teal-500 hover:bg-teal-600">
              <Send className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </CardContent>
    </Card>
  )
}
