import { Card, CardContent, CardFooter, CardHeader } from "@/components/ui/card"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Button } from "@/components/ui/button"
import { Heart, MessageCircle, Share2 } from "lucide-react"
import Image from "next/image"
import Link from "next/link"

interface IdeaCardProps {
  idea: {
    id: number
    title: string
    description: string
    tags: string[]
    author: string
    avatar: string
    image: string
    likes: number
    comments: number
    createdAt: string
  }
}

export function IdeaCard({ idea }: IdeaCardProps) {
  return (
    <Card className="overflow-hidden hover:shadow-lg transition-shadow">
      <div className="relative h-48">
        <Image src={idea.image || "/placeholder.svg"} alt={idea.title} fill className="object-cover" />
      </div>
      <CardHeader className="pb-2">
        <div className="flex items-center gap-3 mb-2">
          <Avatar className="h-8 w-8">
            <AvatarImage src={idea.avatar || "/placeholder.svg"} alt={idea.author} />
            <AvatarFallback>{idea.author.charAt(0)}</AvatarFallback>
          </Avatar>
          <div className="flex flex-col">
            <span className="text-sm font-medium">{idea.author}</span>
            <span className="text-xs text-gray-500">{idea.createdAt}</span>
          </div>
        </div>
        <Link href={`/idea-board/${idea.id}`} className="hover:underline">
          <h3 className="font-bold text-lg">{idea.title}</h3>
        </Link>
      </CardHeader>
      <CardContent className="pb-2">
        <p className="text-gray-600 text-sm line-clamp-2">{idea.description}</p>
        <div className="flex flex-wrap gap-2 mt-3">
          {idea.tags.map((tag, index) => (
            <span key={index} className="text-xs bg-teal-100 text-teal-800 px-2 py-1 rounded-full">
              {tag}
            </span>
          ))}
        </div>
      </CardContent>
      <CardFooter className="border-t pt-3">
        <div className="flex justify-between w-full">
          <Button variant="ghost" size="sm" className="text-gray-600 hover:text-teal-600">
            <Heart className="h-4 w-4 mr-1" />
            {idea.likes}
          </Button>
          <Button variant="ghost" size="sm" className="text-gray-600 hover:text-teal-600">
            <MessageCircle className="h-4 w-4 mr-1" />
            {idea.comments}
          </Button>
          <Button variant="ghost" size="sm" className="text-gray-600 hover:text-teal-600">
            <Share2 className="h-4 w-4 mr-1" />
            Share
          </Button>
        </div>
      </CardFooter>
    </Card>
  )
}
